<?php
// v1/items.php
require_once __DIR__.'/db.php';
require_once __DIR__.'/utils.php';

function list_items() {
  $pdo = get_pdo();

  $type = qp('type', 'game');                           // game|app|mod
  $category = qp('category', null);                     // slug
  $q = qp('q', null);
  $page = max(1, int_qp('page', 1));
  $pageSize = min(60, max(1, int_qp('page_size', 24)));
  $offset = ($page-1)*$pageSize;
  $sort = qp('sort','new');                             // new|rating|downloads

  $where = ["i.published = 1"];
  $params = [];

  if (in_array($type, ['game','app','mod'])) {
    $where[] = "i.type = ?";
    $params[] = $type;
  }

  $joinCategory = '';
  if ($category) {
    $joinCategory = "LEFT JOIN item_categories ic ON ic.item_id = i.id
                     LEFT JOIN categories c ON c.id = ic.category_id";
    $where[] = "c.slug = ?";
    $params[] = $category;
  }

  $searchSql = '';
  if ($q) {
    // Try FULLTEXT first, fallback to LIKE
    $searchSql = " AND (MATCH(i.title, i.description) AGAINST(? IN NATURAL LANGUAGE MODE)
                   OR i.title LIKE ? OR i.description LIKE ?)";
    $params[] = $q;
    $like = '%'.sanitize_like($q).'%';
    $params[] = $like; $params[] = $like;
  }

  $orderBy = "i.created_at DESC";
  if ($sort === 'rating') $orderBy = "COALESCE(s.avg_stars,0) DESC, s.total_ratings DESC";
  if ($sort === 'downloads') $orderBy = "(SELECT COUNT(1) FROM downloads d WHERE d.item_id=i.id) DESC, i.created_at DESC";

  $sql = "SELECT SQL_CALC_FOUND_ROWS
            i.id, i.title, i.slug, i.type, i.logo_url, i.version, i.size_bytes,
            COALESCE(s.avg_stars,0) AS avg_stars, COALESCE(s.total_ratings,0) AS total_ratings
          FROM items i
          LEFT JOIN item_rating_stats s ON s.item_id = i.id
          $joinCategory
          WHERE ".implode(' AND ', $where)." $searchSql
          GROUP BY i.id
          ORDER BY $orderBy
          LIMIT $offset, $pageSize";

  $stmt = $pdo->prepare($sql);
  $stmt->execute($params);
  $items = $stmt->fetchAll();

  $total = (int)$pdo->query("SELECT FOUND_ROWS() AS t")->fetch()['t'];

  json_out(['page'=>$page,'page_size'=>$pageSize,'total'=>$total,'items'=>$items]);
}

function item_detail($id) {
  $pdo = get_pdo();
  // Item
  $stmt = $pdo->prepare("SELECT id, title, slug, type, description, logo_url, download_url, sha256,
                                version, size_bytes, developer_name
                         FROM items WHERE id=? AND published=1 LIMIT 1");
  $stmt->execute([(int)$id]);
  $item = $stmt->fetch();
  if (!$item) bad('Item not found', 404);

  // Rating summary
  $rs = $pdo->prepare("SELECT avg_stars, total_ratings, stars_1, stars_2, stars_3, stars_4, stars_5
                       FROM item_rating_stats WHERE item_id=?");
  $rs->execute([(int)$id]);
  $r = $rs->fetch() ?: [
    'avg_stars'=>0,'total_ratings'=>0,'stars_1'=>0,'stars_2'=>0,'stars_3'=>0,'stars_4'=>0,'stars_5'=>0
  ];
  $rating = [
    'avg_stars'=>(float)$r['avg_stars'],
    'total_ratings'=>(int)$r['total_ratings'],
    'histogram'=>[
      '1'=>(int)$r['stars_1'],'2'=>(int)$r['stars_2'],'3'=>(int)$r['stars_3'],'4'=>(int)$r['stars_4'],'5'=>(int)$r['stars_5']
    ]
  ];

  // Screenshots
  $ss = $pdo->prepare("SELECT url FROM item_screenshots WHERE item_id=? ORDER BY sort_order ASC, id ASC");
  $ss->execute([(int)$id]);
  $screens = array_map(fn($row)=>$row['url'], $ss->fetchAll());

  // Similar (same type, share a category)
  $sim = $pdo->prepare("
    SELECT DISTINCT i2.id, i2.title, i2.logo_url
    FROM item_categories ic1
    JOIN item_categories ic2 ON ic2.category_id = ic1.category_id
    JOIN items i2 ON i2.id = ic2.item_id
    WHERE ic1.item_id = ? AND i2.id <> ? AND i2.published=1 AND i2.type=(SELECT type FROM items WHERE id=?)
    ORDER BY RAND() LIMIT 5
  ");
  $sim->execute([(int)$id, (int)$id, (int)$id]);
  $similar = $sim->fetchAll();

  json_out([
    'item'=>$item,
    'rating'=>$rating,
    'screenshots'=>$screens,
    'similar'=>$similar
  ]);
}
